<?php

namespace GrUpnPf\Pro;

if (!defined('ABSPATH')) {
    exit;
}

class GrUpnPf_Pro_Features
{
    private $upn_image_path = '';
    private static $current_email = null;

    // NEW: guards to prevent duplicates
    private static $email_rendered_once = false;
    private static $embed_registered_once = false;

    public function __construct()
    {
        /**
         * === ORDER CREATION / IMAGE GENERATION ===
         */
        add_action('woocommerce_checkout_order_processed', [$this, 'generate_and_save_upn_image_on_checkout'], 10, 1);
        add_action('woocommerce_checkout_update_order_meta', [$this, 'generate_and_save_upn_image_on_checkout'], 10, 2);
        add_action('woocommerce_new_order', [$this, 'generate_and_save_upn_image_on_checkout'], 10, 1);
        add_action('woocommerce_store_api_checkout_order_processed', [$this, 'generate_and_save_upn_image_on_checkout'], 10, 2);
        add_action('woocommerce_order_status_pending_to_on-hold', [$this, 'generate_and_save_upn_image_on_checkout'], 10, 1);

        /**
         * === CAPTURE WC EMAIL OBJECT (critical for WC 8+) ===
         */
        add_action('woocommerce_email', [$this, 'capture_current_email'], 5);                 // legacy fallback
        add_action('woocommerce_email_init', [$this, 'capture_current_email'], 5);            // WC 3–7
        add_action('woocommerce_email_before_send', [$this, 'capture_current_email'], 5, 2);  // WC 8+

        /**
         * === INLINE DISPLAY + PREPARE UPN ===
         */
        add_action('woocommerce_email_before_order_table', [$this, 'prepareUPNForEmail'], 1, 4);
        add_action('woocommerce_email_after_order_table', [$this, 'displayUPNOnEmails'], 20, 4);

        add_action('woocommerce_thankyou', [$this, 'displayUPNOnOrderPage'], 5, 1);

        /**
         * === ATTACHMENT + EMBED ===
         */
        add_filter('woocommerce_email_attachments', [$this, 'add_upn_attachment'], 10, 4);
        add_action('phpmailer_init', [$this, 'embed_upn_image']);

        /**
         * === CLEANUP ===
         */
        add_action('woocommerce_order_status_completed', [$this, 'cleanup_upn_image']);
        add_action('woocommerce_order_status_cancelled', [$this, 'cleanup_upn_image']);
        add_action('woocommerce_order_status_refunded', [$this, 'cleanup_upn_image']);
    }

    /**
     * Capture WC email object
     */
    public function capture_current_email($email)
    {
        if (is_array($email))
            return;

        if ($email instanceof \WC_Email) {
            self::$current_email = $email;
            \GrUpnPf\grupnpf_log("PRO: capture_current_email - Current email ID: " . $email->id);
        } else {
            \GrUpnPf\grupnpf_log("PRO: capture_current_email - Ignored non-WC_Email object");
        }
    }

    /**
     * Generate & save image on checkout
     */
    public function generate_and_save_upn_image_on_checkout($order_id)
    {
        \GrUpnPf\grupnpf_log('PRO: Hook fired for order ' . $order_id);
        $order = wc_get_order($order_id);

        if (!$order || $order->get_payment_method() !== 'bacs') {
            \GrUpnPf\grupnpf_log('PRO: Exiting generation - not BACS.');
            return;
        }

        $existing_path = get_post_meta($order_id, '_grupnpf_upn_path', true);
        if ($existing_path && file_exists($existing_path)) {
            return;
        }

        $limit = \GrUpnPf\Pro\GrUpnPf_Pro::get_usage_limitations();
        $usage = \GrUpnPf\Pro\GrUpnPf_Pro::get_current_usage();

        if ($limit !== -1 && $usage >= $limit) {
            \GrUpnPf\grupnpf_log('PRO: Usage limit reached.');
            return;
        }

        $logic = new \GrUpnPf\GrUpnPf_Logic_UPN();
        $png = $logic->genUPN($order);

        if ($png) {
            $image_path = $logic->saveUpnImage($png, $order_id);
            update_post_meta($order_id, '_grupnpf_upn_path', $image_path);
            \GrUpnPf\Pro\GrUpnPf_Pro::log_usage($order_id);
            \GrUpnPf\grupnpf_log("PRO: Saved image to {$image_path}");
        }
    }

    /**
     * Attach UPN file to on-hold email
     */
    public function add_upn_attachment($attachments, $email_id, $order, $email = null)
    {
        \GrUpnPf\grupnpf_log("PRO: add_upn_attachment triggered for {$email_id}");

        if (!is_a($order, 'WC_Order') || $order->get_payment_method() !== 'bacs') {
            return $attachments;
        }

        if ($email_id !== 'customer_on_hold_order') {
            return $attachments;
        }

        $image_path = get_post_meta($order->get_id(), '_grupnpf_upn_path', true);

        if (
            get_option('grupnpf_email_priponka') === 'yes' &&
            $image_path && file_exists($image_path)
        ) {

            $attachments[] = $image_path;
            \GrUpnPf\grupnpf_log("PRO: Attached UPN image {$image_path}");
        }

        return $attachments;
    }

    /**
     * Called early in email template — sets correct image path
     */
    public function prepareUPNForEmail($order, $sentToAdmin, $plainText, $email = null)
    {
        // Start local buffer — ONLY for our own output
        ob_start();

        \GrUpnPf\grupnpf_log("PRO: prepareUPNForEmail triggered.");

        if ($sentToAdmin || $plainText)
            return;
        if (!$email || $email->id !== 'customer_on_hold_order')
            return;
        if ($order->get_payment_method() !== 'bacs')
            return;

        $this->upn_image_path = get_post_meta($order->get_id(), '_grupnpf_upn_path', true);

        if ($this->upn_image_path && file_exists($this->upn_image_path)) {
            \GrUpnPf\grupnpf_log("PRO: prepareUPNForEmail - Path set: {$this->upn_image_path}");
        }

        // Flush our buffer safely (no suppression!)
        echo ob_get_clean();
    }


    /**
     * Inline HTML output + register embed callback (only once)
     */
    public function displayUPNOnEmails($order, $sentToAdmin, $plainText, $email = null)
    {
        // Start local buffer — captures ONLY our own HTML
        ob_start();

        \GrUpnPf\grupnpf_log("PRO: displayUPNOnEmails triggered.");

        if ($sentToAdmin || $plainText)
            return;
        if (!$email || $email->id !== 'customer_on_hold_order')
            return;
        if ($order->get_payment_method() !== 'bacs')
            return;
        if (get_option('grupnpf_email_vstavi_upn') !== 'yes')
            return;

        $path = get_post_meta($order->get_id(), '_grupnpf_upn_path', true);
        if (!$path || !file_exists($path)) {
            \GrUpnPf\grupnpf_log("PRO: displayUPNOnEmails - No valid UPN path.");
            return;
        }

        $cid = basename($path);

        // --- PREVENT DUPLICATE HTML ---
        if (!self::$email_rendered_once) {

            self::$email_rendered_once = true;

            echo '<div style="margin-top:20px;">';
            echo '<img src="cid:' . esc_attr($cid) . '" alt="UPN QR" />';
            echo '</div>';

            \GrUpnPf\grupnpf_log("PRO: displayUPNOnEmails - HTML printed CID {$cid}");
        } else {
            \GrUpnPf\grupnpf_log("PRO: displayUPNOnEmails - Skipped duplicate render.");
        }

        // --- REGISTER EMBED ONLY ONCE ---
        if (!self::$embed_registered_once) {

            self::$embed_registered_once = true;

            add_action('phpmailer_init', function ($phpmailer) use ($path, $cid) {
                try {
                    $phpmailer->AddEmbeddedImage($path, $cid, $cid, 'base64', 'image/png');
                    \GrUpnPf\grupnpf_log("PRO: Inline embedded CID {$cid}");
                } catch (\Exception $e) {
                    \GrUpnPf\grupnpf_log("PRO: Inline embed failed: " . $e->getMessage());
                }
            });

        } else {
            \GrUpnPf\grupnpf_log("PRO: embed already registered — skipped duplicate.");
        }

        // Output OUR buffer (never suppress!)
        echo ob_get_clean();
    }


    /**
     * Fallback embed (kept for compatibility, skipped if inline already added)
     */
    public function embed_upn_image($phpmailer)
    {
        \GrUpnPf\grupnpf_log("PRO: embed_upn_image triggered.");

        if (self::$embed_registered_once) {
            \GrUpnPf\grupnpf_log("PRO: embed_upn_image - Skipped because inline already registered.");
            return;
        }
    }

    /**
     * Thank-you page display
     */
    public function displayUPNOnOrderPage($orderId)
    {
        if (\GrUpnPf\GrUpnPf_Logic_UPN::$upn_thankyou_displayed)
            return;

        $order = wc_get_order($orderId);
        if (!$order || $order->get_payment_method() !== 'bacs')
            return;

        $image_path = get_post_meta($orderId, '_grupnpf_upn_path', true);
        if ($image_path && file_exists($image_path)) {

            $img = base64_encode(file_get_contents($image_path));
            echo '<div class="grupnpf-upn">';
            echo '<img src="data:image/png;base64,' . $img . '" alt="UPN QR" />';
            echo '</div>';

            \GrUpnPf\GrUpnPf_Logic_UPN::$upn_thankyou_displayed = true;
        }
    }

    /**
     * Cleanup image when order completes/cancels/refunds
     */
    public function cleanup_upn_image($order)
    {
        $order_id = is_object($order) ? $order->get_id() : $order;
        $file_path = get_post_meta($order_id, '_grupnpf_upn_path', true);

        if ($file_path && file_exists($file_path) && wp_is_writable($file_path)) {
            wp_delete_file($file_path);
        }

        delete_post_meta($order_id, '_grupnpf_upn_path');
    }
}

new GrUpnPf_Pro_Features();