<?php
namespace grUPN;

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Plugin Name: GR UPN payment form
 * Description: Generates UPN payment forms (Slovenian) with QR codes for customers.
 * Version: 1.0
 * Author: Grega Rednak
 * Author URI: https://gregarednak.si
 * Text Domain: gr-upn-payment-form
 * Domain Path: /languages
 * Tested up to: 6.8
 * Stable tag: 1.0
 * Requires PHP: 8.2
 * Tags: woocommerce, qr, upn, slovenia, bacs
 * License: GPL-3.0
 * License URI: https://www.gnu.org/licenses/gpl-3.0.html
 */

if ( ! function_exists( 'grfreemium' ) ) {
    // Create a helper function for easy SDK access.
    function grfreemium() {
        global $grfreemium;

        if ( ! isset( $grfreemium ) ) {
            // Include Freemius SDK.
            require_once dirname( __FILE__ ) . '/vendor/freemius/start.php';
            $grfreemium = fs_dynamic_init( array(
                'id'                  => '14756',
                'slug'                => 'gr-upn-payment-form',
                'type'                => 'plugin',
                'public_key'          => 'pk_82748159feee520dfb4bf6fec3797',
                'is_premium'          => true,
                'premium_suffix'      => 'Pro',
                // If your plugin is a serviceware, set this option to false.
                'has_premium_version' => true,
                'has_addons'          => false,
                'has_paid_plans'      => true,
                'is_org_compliant'    => false,
                'trial'               => array(
                    'days'               => 3,
                    'is_require_payment' => true,
                ),
                'menu'                => array(
                    'first-path'     => 'plugins.php',
                    'contact'        => false,
                    'support'        => false,
                ),
            ) );
        }

        return $grfreemium;
    }

    // Init Freemius.
    grfreemium();
    // Signal that SDK was initiated.
    do_action( 'grfreemium_loaded' );
}

require_once plugin_dir_path(__FILE__) . 'vendor/autoload.php';

grfreemium()->add_action('after_uninstall', 'grUPN\\uninstall_cleanup');
function uninstall_cleanup()
{
    $options = [
        'gr_namen_placila',
        'gr_koda_namena',
        'gr_iban_prejemnika',
        'gr_referenca_prejemnika',
        'gr_ime_prejemnika',
        'gr_ulica_prejemnika',
        'gr_kraj_prejemnika',
        'gr_posta_prejemnika',
        'gr_email_priponka',
        'gr_email_vstavi_upn'
    ];
    foreach ($options as $option) {
        delete_option($option);
    }
}

add_action('woocommerce_init', 'grUPN\\grUPN_init', 20);
function grUPN_init()
{
    new LogicUPN();
}

add_action('wp_ajax_send_upn_to_customer', 'grUPN\\gr_upn_ajax_send_upn_to_customer');
function gr_upn_ajax_send_upn_to_customer()
{
	$nonce = isset($_POST['nonce']) ? sanitize_text_field(wp_unslash($_POST['nonce'])) : '';
	
	if (empty($nonce) || !wp_verify_nonce($nonce, 'gr_send_upn_nonce')) {
		wp_send_json_error(__('Neveljaven zahtevek.', 'gr-upn-payment-form'));
		wp_die();
	}
	
    $order_id = isset($_POST['order_id']) ? intval($_POST['order_id']) : 0;
    if ($order_id > 0) {
        $order = wc_get_order($order_id);
        if ($order && current_user_can('edit_shop_orders')) {
            gr_upn_send_upn_email($order_id);
            wp_send_json_success(__('UPN uspešno poslan.', 'gr-upn-payment-form'));
        }
    }

    wp_send_json_error(__('Napaka pri pošiljanju UPN.', 'gr-upn-payment-form'));
    wp_die();
}

function gr_upn_send_upn_email($order_id)
{
    $order = wc_get_order($order_id);
    if ('bacs' === $order->get_payment_method()) {
        $UPN = new LogicUPN();
        $png = $UPN->genUPN($order);
        $upn_image_path = $UPN->saveUpnImage($png, $order->get_id());

        if ($upn_image_path) {
            $subject = __('UPN obrazec za naročilo #', 'gr-upn-payment-form') . $order_id;
            $message = __('V prilogi imate UPN obrazec za lažje plačilo na bančni račun.', 'gr-upn-payment-form');
            $headers = ['Content-Type: text/html; charset=UTF-8'];
            $recipient = $order->get_billing_email();
            $attachments = [$upn_image_path];
            wp_mail($recipient, $subject, $message, $headers, $attachments);
            if (file_exists($upn_image_path)) {
                wp_delete_file($upn_image_path);
            }
        }
    }
}

if (is_admin()) {
    add_action('admin_init', function () {
        if (wp_doing_ajax()) return;
        if (!in_array('woocommerce/woocommerce.php', apply_filters('active_plugins', get_option('active_plugins')))) {
            echo '<div class="notice notice-error"><p>' .
                esc_html__('GR UPN plačilni nalog - QR vtičnik potrebuje WooCommerce za delovanje!', 'gr-upn-payment-form') .
                '</p></div>';
        }
        if (grfreemium()->is_not_paying()) {
            echo '<div class="notice notice-warning is-dismissible"><p>' .
                '<a href="' . esc_url(grfreemium()->get_upgrade_url()) . '">' .
                esc_html__('Nagradite za avtomatsko generiranje in prikazovanje UPN obrazcev.', 'gr-upn-payment-form') .
                '</a></p></div>';
        }
    });

    function grUPN_add_upn_settings($settings)
    {
        if (class_exists('WC_Settings_Page')) {
            $settings[] = new WC_Settings_UPN();
        }
        return $settings;
    }
    add_filter('woocommerce_get_settings_pages', 'grUPN\\grUPN_add_upn_settings');

    function gr_upn_enqueue_scripts($hook)
    {
        $css_path = plugin_dir_path(__FILE__) . 'src/assets/css/main.css';
		$css_version = file_exists($css_path) ? filemtime($css_path) : '1.0.0';

		wp_register_style(
			'main_css',
			plugin_dir_url(__FILE__) . 'src/assets/css/main.css',
			[],
			$css_version
		);
        wp_enqueue_style('main_css');
        wp_enqueue_script(
            'gr_upn_admin_script',
            plugin_dir_url(__FILE__) . 'src/assets/js/admin.js',
            ['jquery'],
            '1.0.0',
            true
        );
        wp_localize_script('gr_upn_admin_script', 'grUPNAjax', ['ajaxurl' => admin_url('admin-ajax.php')]);
    }
    add_action('admin_enqueue_scripts', 'grUPN\\gr_upn_enqueue_scripts');

    $plugin = plugin_basename(__FILE__);
    add_filter("plugin_action_links_$plugin", 'grUPN\\add_plugin_settings_link');
    function add_plugin_settings_link($links)
    {
        $links[] = '<a href="admin.php?page=wc-settings&tab=upn_settings">' . esc_html__('Nastavitve', 'gr-upn-payment-form') . '</a>';
        $links[] = '<a href="' . esc_url(grfreemium()->get_upgrade_url()) . '">' . esc_html__('Nagradite', 'gr-upn-payment-form') . '</a>';
        return $links;
    }

    add_filter('plugin_row_meta', 'grUPN\\add_plugin_row_meta', 10, 2);
    function add_plugin_row_meta($plugin_meta, $plugin_file)
    {
        if (strpos($plugin_file, 'gr-upn.php') !== false) {
            $plugin_meta[] = '<a href="admin.php?page=wc-settings&tab=upn_settings">' . esc_html__('Nastavitve', 'gr-upn-payment-form') . '</a>';
            $plugin_meta[] = '<a href="' . esc_url(grfreemium()->get_upgrade_url()) . '">' . esc_html__('Nagradite', 'gr-upn-payment-form') . '</a>';
        }
        return $plugin_meta;
    }
}

if (!wp_next_scheduled('gr_upn_daily_cleanup')) {
    wp_schedule_event(time(), 'daily', 'gr_upn_daily_cleanup');
}
add_action('gr_upn_daily_cleanup', 'gr_upn_cleanup_images');
function gr_upn_cleanup_images()
{
    $upload_dir = wp_upload_dir();
    $dir = $upload_dir['path'];
    $files = glob($dir . '/gr-upn-order-*.png');
    foreach ($files as $file) {
        if (is_file($file) && time() - filemtime($file) > 86400) {
            wp_delete_file($file);
        }
    }
}