<?php
namespace grUPN;

if (!defined('ABSPATH')) exit;

use InvalidArgumentException;
use Endroid\QrCode\Builder\Builder;
use Endroid\QrCode\Encoding\Encoding;
use Endroid\QrCode\ErrorCorrectionLevel;
use Endroid\QrCode\Writer\PngWriter;
use Endroid\QrCode\RoundBlockSizeMode;

class UPN
{
    private string $payerName;
    private string $payerAddress;
    private string $payerPost;

    private string $receiverName;
    private string $receiverAddress;
    private string $receiverPost;
    private string $receiverIban;
    private string $reference;

    private float $amount;
    private string $paymentCode;
    private string $paymentPurpose = "";
    private \DateTime $dueDate;

    private const FONT = __DIR__ . "/assets/fonts/courier-new-bold.ttf";
    private const FONT_SIZE = 17;
    private const FONT_SMALL = 12;

    private $image;
    private $color;

    public function __construct()
    {
        $this->image = imagecreatefrompng(__DIR__ . "/assets/media/upn.png");
        $this->color = imagecolorallocate($this->image, 0x00, 0x00, 0x00);
    }

    public function gdResource()
    {
        $this->writeText(697, 170, esc_html($this->payerName ?? ""));
        $this->writeText(697, 201, esc_html($this->payerAddress ?? ""));
        $this->writeText(697, 233, esc_html($this->payerPost ?? ""));

        $this->writeText(30, 62, esc_html($this->payerName ?? ""), self::FONT_SMALL);
        $this->writeText(30, 87, esc_html($this->payerAddress ?? ""), self::FONT_SMALL);
        $this->writeText(30, 112, esc_html($this->payerPost ?? ""), self::FONT_SMALL);

        $this->writeText(418, 507, esc_html($this->receiverName ?? ""));
        $this->writeText(418, 538, esc_html($this->receiverAddress ?? ""));
        $this->writeText(418, 570, esc_html($this->receiverPost ?? ""));

        $this->writeText(30, 405, esc_html($this->receiverName ?? ""), self::FONT_SMALL);
        $this->writeText(30, 430, esc_html($this->receiverAddress ?? ""), self::FONT_SMALL);
        $this->writeText(30, 455, esc_html($this->receiverPost ?? ""), self::FONT_SMALL);

        $this->writeText(418, 400, esc_html($this->getFormattedReceiverIban() ?? ""));
        $this->writeText(30, 300, esc_html($this->getFormattedReceiverIban() ?? ""), self::FONT_SMALL);

        $this->writeText(418, 451, esc_html($this->getReferencePrefix()));
        $this->writeText(528, 451, esc_html($this->getReferenceSuffix()));
        $this->writeText(30, 351, esc_html($this->getFormattedReference()), self::FONT_SMALL);

        $this->writeText(528, 340, esc_html($this->paymentPurpose ?? ""));
        $this->writeText(30, 165, esc_html($this->paymentPurpose ?? ""), 10);

        $this->writeText(1155, 340, esc_html($this->dueDate->format("d.m.Y")));
        $this->writeText(30, 195, esc_html($this->dueDate->format("d.m.Y")), self::FONT_SMALL);

        $this->writeText(110, 247, esc_html("***" . $this->getFormattedPrice()), self::FONT_SMALL);
        $this->writeText(750, 285, esc_html("***" . $this->getFormattedPrice()));

        $this->writeText(418, 340, esc_html($this->paymentCode ?? ""));

        $qr = $this->getQRCode();
        imagecopyresampled($this->image, $qr, 433, 60, 0, 0, 220, 220, imagesx($qr), imagesy($qr));

        return $this->image;
    }

    public function png(): string
    {
        $image = $this->gdResource();

        ob_start();
        imagepng($image);
        $img = ob_get_contents();
        ob_end_clean();

        imagedestroy($image);

        return $img;
    }

    public function getQRCode()
    {
        $qrText = $this->getQRCodeText();

		$builder = new Builder(
			writer: new PngWriter(),
			data: $qrText,
			encoding: new Encoding('UTF-8'),
			errorCorrectionLevel: ErrorCorrectionLevel::High,
			size: 400,
			margin: 0,
			roundBlockSizeMode: RoundBlockSizeMode::Margin
		);

		$result = $builder->build();

		return imagecreatefromstring($result->getString());
    }

    public function getQRCodeText(): string
    {
        $text = [
            "UPNQR",
            "",
            "    ",
            "",
            "",
            $this->payerName ?? "",
            $this->payerAddress ?? "",
            $this->payerPost ?? "",
            sprintf("%011d", $this->amount * 100) ?? "",
            "",
            "",
            $this->paymentCode ?? "",
            $this->paymentPurpose ?? "",
            $this->dueDate->format("d.m.Y") ?? "",
            $this->receiverIban ?? "",
            $this->reference ?? "",
            $this->receiverName ?? "",
            $this->receiverAddress ?? "",
            $this->receiverPost ?? "",
        ];

        array_walk($text, fn(&$i) => ($i = trim($i)));
        $text = implode("\n", $text) . "\n";
        $text .= mb_strlen($text) . "\n";

        return $text;
    }

    public function getFormattedPrice(): string
    {
        return number_format($this->amount, 2, ",", ".");
    }

    public function getFormattedReceiverIban(): string
    {
        return wordwrap($this->receiverIban, 4, " ", true);
    }

    public function getFormattedReference(): string
    {
        return $this->getReferencePrefix() . " " . $this->getReferenceSuffix();
    }

    public function getReferencePrefix(): string
    {
        return substr($this->reference, 0, 4);
    }

    public function getReferenceSuffix(): string
    {
        return substr($this->reference, 4);
    }

    private function writeText(int $x, int $y, $text, $fontSize = self::FONT_SIZE): void
    {
        if ($text) {
            imagefttext($this->image, $fontSize, 0, $x, $y, $this->color, self::FONT, $text);
        }
    }

    public function setPayerName(string $name): self
    {
        $this->payerName = mb_substr($name, 0, 33);
        return $this;
    }

    public function setPayerAddress(string $address): self
    {
        $this->payerAddress = mb_substr($address, 0, 33);
        return $this;
    }

    public function setPayerPost(string $post): self
    {
        $this->payerPost = mb_substr($post, 0, 33);
        return $this;
    }

    public function setReceiverName(string $name): self
    {
        $this->receiverName = mb_substr($name, 0, 33);
        return $this;
    }

    public function setReceiverAddress(string $address): self
    {
        $this->receiverAddress = mb_substr($address, 0, 33);
        return $this;
    }

    public function setReceiverPost(string $post): self
    {
        $this->receiverPost = mb_substr($post, 0, 33);
        return $this;
    }

    public function setReceiverIban(string $iban): self
    {
        $iban = str_replace(" ", "", $iban);
        if (strlen($iban) > 33) {
            throw new InvalidArgumentException("Najdaljši IBAN je 33 znakov");
        }
        $this->receiverIban = $iban;
        return $this;
    }

    public function setReference(string $reference): self
    {
        $reference = str_replace(" ", "", $reference);
        if (strlen($reference) > 26) {
            throw new InvalidArgumentException("Najdaljša referenca je 26 znakov");
        }
        $this->reference = $reference;
        return $this;
    }

    public function setAmount(float $amount): self
    {
        $this->amount = $amount;
        return $this;
    }

    public function setCode(string $code): self
    {
        if (strlen($code) !== 4) {
            throw new InvalidArgumentException("Koda namena mora biti 4 črke");
        }
        $this->paymentCode = strtoupper($code);
        return $this;
    }

    public function setPurpose(string $purpose): self
    {
        $this->paymentPurpose = mb_substr($purpose, 0, 42);
        return $this;
    }

    public function setDueDate(\DateTime $date): self
    {
        $this->dueDate = $date;
        return $this;
    }
}